/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math3.linear;

import org.apache.commons.math3.exception.MaxCountExceededException;
import org.apache.commons.math3.exception.util.Localizable;
import org.apache.commons.math3.exception.util.LocalizedFormats;
import org.apache.commons.math3.linear.HessenbergTransformer;
import org.apache.commons.math3.linear.MatrixUtils;
import org.apache.commons.math3.linear.NonSquareMatrixException;
import org.apache.commons.math3.linear.RealMatrix;
import org.apache.commons.math3.util.FastMath;
import org.apache.commons.math3.util.Precision;

class SchurTransformer {
    private static final int MAX_ITERATIONS = 100;
    private final double[][] matrixP;
    private final double[][] matrixT;
    private RealMatrix cachedP;
    private RealMatrix cachedT;
    private RealMatrix cachedPt;
    private final double epsilon = Precision.EPSILON;

    public SchurTransformer(RealMatrix matrix) {
        if (!matrix.isSquare()) {
            throw new NonSquareMatrixException(matrix.getRowDimension(), matrix.getColumnDimension());
        }
        HessenbergTransformer transformer = new HessenbergTransformer(matrix);
        this.matrixT = transformer.getH().getData();
        this.matrixP = transformer.getP().getData();
        this.cachedT = null;
        this.cachedP = null;
        this.cachedPt = null;
        this.transform();
    }

    public RealMatrix getP() {
        if (this.cachedP == null) {
            this.cachedP = MatrixUtils.createRealMatrix(this.matrixP);
        }
        return this.cachedP;
    }

    public RealMatrix getPT() {
        if (this.cachedPt == null) {
            this.cachedPt = this.getP().transpose();
        }
        return this.cachedPt;
    }

    public RealMatrix getT() {
        if (this.cachedT == null) {
            this.cachedT = MatrixUtils.createRealMatrix(this.matrixT);
        }
        return this.cachedT;
    }

    private void transform() {
        int n2 = this.matrixT.length;
        double norm = this.getNorm();
        ShiftInfo shift = new ShiftInfo();
        int iteration = 0;
        int iu = n2 - 1;
        while (iu >= 0) {
            int il = this.findSmallSubDiagonalElement(iu, norm);
            if (il == iu) {
                this.matrixT[iu][iu] = this.matrixT[iu][iu] + shift.exShift;
                --iu;
                iteration = 0;
                continue;
            }
            if (il == iu - 1) {
                double p = (this.matrixT[iu - 1][iu - 1] - this.matrixT[iu][iu]) / 2.0;
                double q = p * p + this.matrixT[iu][iu - 1] * this.matrixT[iu - 1][iu];
                double[] dArray = this.matrixT[iu];
                int n3 = iu;
                dArray[n3] = dArray[n3] + shift.exShift;
                double[] dArray2 = this.matrixT[iu - 1];
                int n4 = iu - 1;
                dArray2[n4] = dArray2[n4] + shift.exShift;
                if (q >= 0.0) {
                    int i2;
                    double z = FastMath.sqrt(FastMath.abs(q));
                    z = p >= 0.0 ? p + z : p - z;
                    double x = this.matrixT[iu][iu - 1];
                    double s = FastMath.abs(x) + FastMath.abs(z);
                    p = x / s;
                    q = z / s;
                    double r = FastMath.sqrt(p * p + q * q);
                    p /= r;
                    q /= r;
                    for (int j2 = iu - 1; j2 < n2; ++j2) {
                        z = this.matrixT[iu - 1][j2];
                        this.matrixT[iu - 1][j2] = q * z + p * this.matrixT[iu][j2];
                        this.matrixT[iu][j2] = q * this.matrixT[iu][j2] - p * z;
                    }
                    for (i2 = 0; i2 <= iu; ++i2) {
                        z = this.matrixT[i2][iu - 1];
                        this.matrixT[i2][iu - 1] = q * z + p * this.matrixT[i2][iu];
                        this.matrixT[i2][iu] = q * this.matrixT[i2][iu] - p * z;
                    }
                    for (i2 = 0; i2 <= n2 - 1; ++i2) {
                        z = this.matrixP[i2][iu - 1];
                        this.matrixP[i2][iu - 1] = q * z + p * this.matrixP[i2][iu];
                        this.matrixP[i2][iu] = q * this.matrixP[i2][iu] - p * z;
                    }
                }
                iu -= 2;
                iteration = 0;
                continue;
            }
            this.computeShift(il, iu, iteration, shift);
            if (++iteration > 100) {
                throw new MaxCountExceededException((Localizable)LocalizedFormats.CONVERGENCE_FAILED, 100, new Object[0]);
            }
            double[] hVec = new double[3];
            int im = this.initQRStep(il, iu, shift, hVec);
            this.performDoubleQRStep(il, im, iu, shift, hVec);
        }
    }

    private double getNorm() {
        double norm = 0.0;
        for (int i2 = 0; i2 < this.matrixT.length; ++i2) {
            for (int j2 = FastMath.max(i2 - 1, 0); j2 < this.matrixT.length; ++j2) {
                norm += FastMath.abs(this.matrixT[i2][j2]);
            }
        }
        return norm;
    }

    private int findSmallSubDiagonalElement(int startIdx, double norm) {
        int l2;
        for (l2 = startIdx; l2 > 0; --l2) {
            double s = FastMath.abs(this.matrixT[l2 - 1][l2 - 1]) + FastMath.abs(this.matrixT[l2][l2]);
            if (s == 0.0) {
                s = norm;
            }
            if (FastMath.abs(this.matrixT[l2][l2 - 1]) < this.epsilon * s) break;
        }
        return l2;
    }

    private void computeShift(int l2, int idx, int iteration, ShiftInfo shift) {
        double s;
        shift.x = this.matrixT[idx][idx];
        shift.w = 0.0;
        shift.y = 0.0;
        if (l2 < idx) {
            shift.y = this.matrixT[idx - 1][idx - 1];
            shift.w = this.matrixT[idx][idx - 1] * this.matrixT[idx - 1][idx];
        }
        if (iteration == 10) {
            shift.exShift += shift.x;
            int i2 = 0;
            while (i2 <= idx) {
                double[] dArray = this.matrixT[i2];
                int n2 = i2++;
                dArray[n2] = dArray[n2] - shift.x;
            }
            s = FastMath.abs(this.matrixT[idx][idx - 1]) + FastMath.abs(this.matrixT[idx - 1][idx - 2]);
            shift.x = 0.75 * s;
            shift.y = 0.75 * s;
            shift.w = -0.4375 * s * s;
        }
        if (iteration == 30) {
            s = (shift.y - shift.x) / 2.0;
            if ((s = s * s + shift.w) > 0.0) {
                s = FastMath.sqrt(s);
                if (shift.y < shift.x) {
                    s = -s;
                }
                s = shift.x - shift.w / ((shift.y - shift.x) / 2.0 + s);
                int i3 = 0;
                while (i3 <= idx) {
                    double[] dArray = this.matrixT[i3];
                    int n3 = i3++;
                    dArray[n3] = dArray[n3] - s;
                }
                shift.exShift += s;
                shift.w = 0.964;
                shift.y = 0.964;
                shift.x = 0.964;
            }
        }
    }

    private int initQRStep(int il, int iu, ShiftInfo shift, double[] hVec) {
        int im;
        for (im = iu - 2; im >= il; --im) {
            double rhs;
            double lhs;
            double z = this.matrixT[im][im];
            double r = shift.x - z;
            double s = shift.y - z;
            hVec[0] = (r * s - shift.w) / this.matrixT[im + 1][im] + this.matrixT[im][im + 1];
            hVec[1] = this.matrixT[im + 1][im + 1] - z - r - s;
            hVec[2] = this.matrixT[im + 2][im + 1];
            if (im == il || (lhs = FastMath.abs(this.matrixT[im][im - 1]) * (FastMath.abs(hVec[1]) + FastMath.abs(hVec[2]))) < this.epsilon * (rhs = FastMath.abs(hVec[0]) * (FastMath.abs(this.matrixT[im - 1][im - 1]) + FastMath.abs(z) + FastMath.abs(this.matrixT[im + 1][im + 1])))) break;
        }
        return im;
    }

    private void performDoubleQRStep(int il, int im, int iu, ShiftInfo shift, double[] hVec) {
        int n2 = this.matrixT.length;
        double p = hVec[0];
        double q = hVec[1];
        double r = hVec[2];
        for (int k2 = im; k2 <= iu - 1; ++k2) {
            boolean notlast;
            boolean bl = notlast = k2 != iu - 1;
            if (k2 != im) {
                p = this.matrixT[k2][k2 - 1];
                q = this.matrixT[k2 + 1][k2 - 1];
                r = notlast ? this.matrixT[k2 + 2][k2 - 1] : 0.0;
                shift.x = FastMath.abs(p) + FastMath.abs(q) + FastMath.abs(r);
                if (!Precision.equals(shift.x, 0.0, this.epsilon)) {
                    p /= shift.x;
                    q /= shift.x;
                    r /= shift.x;
                }
            }
            if (shift.x == 0.0) break;
            double s = FastMath.sqrt(p * p + q * q + r * r);
            if (p < 0.0) {
                s = -s;
            }
            if (s == 0.0) continue;
            if (k2 != im) {
                this.matrixT[k2][k2 - 1] = -s * shift.x;
            } else if (il != im) {
                this.matrixT[k2][k2 - 1] = -this.matrixT[k2][k2 - 1];
            }
            shift.x = (p += s) / s;
            shift.y = q / s;
            double z = r / s;
            q /= p;
            r /= p;
            for (int j2 = k2; j2 < n2; ++j2) {
                p = this.matrixT[k2][j2] + q * this.matrixT[k2 + 1][j2];
                if (notlast) {
                    this.matrixT[k2 + 2][j2] = this.matrixT[k2 + 2][j2] - (p += r * this.matrixT[k2 + 2][j2]) * z;
                }
                this.matrixT[k2][j2] = this.matrixT[k2][j2] - p * shift.x;
                this.matrixT[k2 + 1][j2] = this.matrixT[k2 + 1][j2] - p * shift.y;
            }
            for (int i2 = 0; i2 <= FastMath.min(iu, k2 + 3); ++i2) {
                p = shift.x * this.matrixT[i2][k2] + shift.y * this.matrixT[i2][k2 + 1];
                if (notlast) {
                    this.matrixT[i2][k2 + 2] = this.matrixT[i2][k2 + 2] - (p += z * this.matrixT[i2][k2 + 2]) * r;
                }
                this.matrixT[i2][k2] = this.matrixT[i2][k2] - p;
                this.matrixT[i2][k2 + 1] = this.matrixT[i2][k2 + 1] - p * q;
            }
            int high = this.matrixT.length - 1;
            for (int i3 = 0; i3 <= high; ++i3) {
                p = shift.x * this.matrixP[i3][k2] + shift.y * this.matrixP[i3][k2 + 1];
                if (notlast) {
                    this.matrixP[i3][k2 + 2] = this.matrixP[i3][k2 + 2] - (p += z * this.matrixP[i3][k2 + 2]) * r;
                }
                this.matrixP[i3][k2] = this.matrixP[i3][k2] - p;
                this.matrixP[i3][k2 + 1] = this.matrixP[i3][k2 + 1] - p * q;
            }
        }
        for (int i4 = im + 2; i4 <= iu; ++i4) {
            this.matrixT[i4][i4 - 2] = 0.0;
            if (i4 <= im + 2) continue;
            this.matrixT[i4][i4 - 3] = 0.0;
        }
    }

    private static class ShiftInfo {
        double x;
        double y;
        double w;
        double exShift;

        private ShiftInfo() {
        }
    }
}

