/* ***** BEGIN LICENSE BLOCK *****
 * Distributed under the BSD license:
 *
 * Copyright (c) 2010, Ajax.org B.V.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Ajax.org B.V. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL AJAX.ORG B.V. BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * ***** END LICENSE BLOCK ***** */


// HEAVILY MODIFIED

var dom = ace.require("ace/lib/dom");
var oop = ace.require("ace/lib/oop");
var aceEvent = ace.require("ace/lib/event");
var Range = ace.require("ace/range").Range;
var Tooltip = ace.require("ace/tooltip").Tooltip;

function TokenTooltip (editor) {
    if (editor.tokenTooltip)
        return;
    Tooltip.call(this, editor.container);
    editor.tokenTooltip = this;
    this.editor = editor;

    this.update = this.update.bind(this);
    this.onMouseMove = this.onMouseMove.bind(this);
    this.onMouseOut = this.onMouseOut.bind(this);
    aceEvent.addListener(editor.renderer.scroller, "mousemove", this.onMouseMove);
    aceEvent.addListener(editor.renderer.content, "mouseout", this.onMouseOut);
}

oop.inherits(TokenTooltip, Tooltip);

(function(){
    this.token = {};
    this.range = new Range();

    var self = this;

    var xhr = new XMLHttpRequest();
    xhr.onreadystatechange = function() {
        if (xhr.readyState == XMLHttpRequest.DONE) {
            reqToken.text = xhr.responseText;
            self.update.call(fileManager.editor.tokenTooltip);
        }
    }

    var reqToken = {
        start:"",
        value:"",
        text:"",
    };

    function isLetter(c) {
      return c.toLowerCase() != c.toUpperCase();
    }

    this.update = function() {
        // is char a separator?
        function sepChar(c,string) {
            if (string) return false;
            var chars = [" ","(",")","\t",",","#","*","="]; // remove "/" to make it work with ids, right???
            return (chars.indexOf(c) != -1);
        }

        // decode value
        function deco(str) {
            return decodeURIComponent(str);
        }

        function getMyTokens() {
            var ts = session.getTabSize();
            var line = session.getLine(row);
            var tokens = [];
            var token;
            var string = ""; // empty|"|'
            var c;
            var col = 0; // real column cursor
            for (var i=0; i<=line.length; i++) {
                c = line[i];
                // done?
                if (i == line.length || sepChar(c,string)) {
                    if (token) {
                        token.length = token.value.length;
                        token.prev = tokens.length ? tokens.last() : null;
                        //if (token.value.length>1 || isLetter(token.value))
                        tokens.push(token);
                    }
                    // add exceptions:
                    // "(" - we need to detect stuff like "fun ( token"
                    // "," - to detect stuff like "fun ( token, token )"
                    if (c == "(" || c == "," || c == "=") {
                        tokens.push({
                            index: tokens.length,
                            start: i,
                            end: i+1,
                            length: 1,
                            type: "",
                            colStart: col+1,
                            colEnd: col+1,
                            value: c
                        });
                    }
                    token = null;
                } else {
                    if (c == '"' || c == "'") {
                        if (string && string == c)
                        string = "";
                        else
                        string = c;
                    }
                    if (!token) {
                        token = {
                            index: tokens.length,
                            start: i,
                            end: i,
                            type: string ? "string" : "text",
                            colStart: col,
                            value: ""
                        };
                    }
                    token.end = i+1;
                    token.value += c;
                    token.colEnd = col+1;
                }
                col += (c=="\t") ? ts : 1;
            }
            return tokens;
        }

        function getMyTokenAt(row,column,tokens) {
            if (!tokens)
                tokens = getMyTokens(row);
            for (var i = 0; i < tokens.length; i++) {
                if (column >= tokens[i].colStart && column <= tokens[i].colEnd) {
                    return tokens[i];
                    break;
                }
            }
        }

        function isInJS(row, start) {
            // find first {
            var line, col = start-1;
            var ob = 0;
            while (row>=0) {
                line = session.getLine(row);
                if (col == null)
                    col = line.length-1;
                while (col>=0) {
                    if (line[col] == "}")
                        ob--;
                    else if (line[col] == "{")
                        ob++;
                    if (ob>0)
                        return true;
                    col--;
                }
                col = null;
                row--;
            }
            return false;
        }

        function copyToken(token) {
            var r = {};
            for (var i in token)
                r[i] = token[i];
            return r;
        }

        function _getTokenAt(row,column) {
            var type = "";
            var found = true;
            var i;

            var tokens = getMyTokens(row);
            var token = getMyTokenAt(row, column, tokens);
            if (!token)
                return null;

            var inJS = isInJS(row, token.start);

            var i = token.index;
            // javascript?
            /*
                varGet("<var>")
                varSet("<var>")
                ioGet("<io>")
                ioSet("<io>")
                uiSet(<ui id>,<attr>)
                uiGet(<ui id>,<attr>)
                uiSessionGet(<ui id>,<attr>)
                uiSessionSet(<ui id>,<attr>)
             */
            if (inJS) {
                // overed on ui, attr or comma part?
                if (i>3 &&
                    token.type == "string" &&
                    tokens[i-1].value == "," &&
                    tokens[i-2].type == "string" &&
                    tokens[i-3].value == "(" &&
                    tokens[i-4].value.indexOf("ui") == "0") {
                    // treat as if we overed on the id
                    token = copyToken(tokens[i-2]);
                    i = token.index;
                }
                if (i>2 &&
                    token.value == "," &&
                    tokens[i-1].type == "string" &&
                    tokens[i-2].value == "(" &&
                    tokens[i-3].value.indexOf("ui") == "0") {
                    // treat as if we overed on the id
                    token = copyToken(tokens[i-1]);
                    i = token.index;
                }

                if (i>1 && tokens[i-1].value == "(") {
                    switch (tokens[i-2].value) {
                        case "varGet":
                        case "varSet":
                            type = "var";
                            break;
                        case "ioGet":
                        case "ioSet":
                            type = "io";
                            break;
                        case "uiGet":
                        case "uiSet":
                            type = "ui";
                            break;
                        case "uiSessionGet":
                        case "uiSessionSet":
                            type = "uisession";
                            break;
                    }
                }
                // search after
                if (type.indexOf("ui") == 0) {
                    if (token.type == "string" &&
                        tokens.length > i+2 &&
                        tokens[i+1].value == "," &&
                        tokens[i+2].type == "string") {
                        // include attribute
                        token.length += tokens[i+1].length;
                        token.length += tokens[i+2].length;
                        token.end = tokens[i+2].end;
                        token.colEnd = tokens[i+2].colEnd;
                        // remove all ""
                        token.value = token.value.substr(1,token.value.length-2) +
                                    "." + tokens[i+2].value.substr(1,tokens[i+2].value.length-2);
                    } else {
                        type = "";
                    }
                } else if (type) {
                    // remove ""
                    token.value = token.value.substr(1,token.value.length-2);
                }
            }
            // events
            /*
                io <datapoint>
                $<var>
                uiset <ui>
                uisessionset <ui>
             */
            else {
                var tok = "";
                var keywords = [",", "=", ":", "and", "or"];
                // find comma or first token
                if (keywords.indexOf(token.value)==-1 && i>0) {
                    tok = tokens[0].value; // i-1].value; changed!
                    for (var k=i-1; k>=0; k--) { // check if previously there's a valid action
                        if (keywords.indexOf(tokens[k].value)!=-1 ||
                            tokens[k].value.indexOf(":")>0) {
                            tok = k < i - 1  ? tokens[k + 1].value : "";
                            break;
                        }
                    }
                    //if (!tok)
                }
                switch (tok.toLowerCase()) { // is there a previous token? otherwise default (could be variable)
                    case "io":
                        type = "io";
                        break;
                    case "uiset":
                        type = "ui";
                        break;
                    case "uisessionset":
                        type = "uisession";
                        break;
                    default:
                        if (token.type == "text" && token.value[0] == "$")
                            type = "var";
                        break;
                }
                token.req = token.value;
            }


            if (!type)
                return null;

            token.type = type;
            return token;
        }

        this.$timer = null;

        var r = this.editor.renderer;
        if (this.lastT - (r.timeStamp || 0) > 1000) {
            r.rect = null;
            r.timeStamp = this.lastT;
            this.maxHeight = window.innerHeight;
            this.maxWidth = window.innerWidth;
        }

        var canvasPos = r.rect || (r.rect = r.scroller.getBoundingClientRect());
        var offset = (this.x + r.scrollLeft - canvasPos.left - r.$padding) / r.characterWidth;
        var row = Math.floor((this.y + r.scrollTop - canvasPos.top) / r.lineHeight);
        var col = Math.round(offset);

        var screenPos = {row: row, column: col, side: offset - col > 0 ? 1 : -1};
        var session = this.editor.session;
        var docPos = screenPos; //session.screenToDocumentPosition(screenPos.row, screenPos.column);
        var token = session.getTokenAt(docPos.row, docPos.column);

        var token = _getTokenAt(docPos.row, docPos.column);

        if (!token && !session.getLine(docPos.row)) {
            token = null; /*{
                type: "",
                value: "",
                state: session.bgTokenizer.getState(0)
            };
            */
        }
        if (!token) {
            session.removeMarker(this.marker);
            reqToken.value = "";
            this.hide();
            return;
        }

        if (token.start != reqToken.start || token.value != reqToken.value) {
            reqToken.start = token.start;
            reqToken.value = token.value;
            reqToken.text = "";
            var url = "/x/get";
            var value = encodeURIComponent(token.value).replace(/\./g,"%2E")
            switch (token.type) {
                case "var": url += "vars?"+value.toUpperCase(); break
                case "ui":  url += "ui?"+value; break;
                case "uisession": url += "ui?"+value+".latest"; break;
                case "io":  url += "io?"+value; break;
            }
            url += "."+new Date().getTime();

            xhr.open("GET", url, true);
            xhr.send();
            this.hide();
            return;
        }

        /*
        tokenText = token.type;
        if (token.state)
            tokenText += "|" + token.state;
        if (token.merge)
            tokenText += "\n  merge";
        if (token.stateTransitions)
            tokenText += "\n  " + token.stateTransitions.join("\n  ");
        */

        if (this.tokenText != reqToken.text) {

            var json = JSON.parse(reqToken.text);
            var txt = null;
            if (json) {
                if (json.vars) for (var i in json.vars){
                    if (json.vars[i] != null)
                        txt = deco(json.vars[i]);
                }
                if (json.ui) for (var i in json.ui){
                    if (json.ui[i] != null)
                        txt = deco(json.ui[i]);
                }
                if (json.io) for (var i in json.io){
                    if (json.io[i] != null) {
                        txt = deco(json.io[i][1]) + "<br>";
                        txt += "<span style='font-size:80%'>" + json.io[i][0] + "</span>";
                    }
                }
            }
            if (txt == null) {
                txt = "<span style='color:#aaa'>?</span>";
            } else if (txt == "" || txt.trim() == "") {
                txt = "&nbsp;";
            }
            this.setHtml(txt);
            this.width = this.getWidth();
            this.height = this.getHeight();
            this.tokenText = reqToken.text;
        }

        this.show(null, this.x, this.y);

        this.token = token;
        session.removeMarker(this.marker);
        this.range = new Range(docPos.row, token.start, docPos.row, token.end); //+3);
        this.marker = session.addMarker(this.range, "ace_bracket", "text");
    };

    this.onMouseMove = function(e) {
        this.x = e.clientX;
        this.y = e.clientY;
        if (this.isOpen) {
            this.lastT = e.timeStamp;
            this.setPosition(this.x, this.y);
            if (e.which)
                this.onMouseOut();
        }
        if (e.which)
            return;
        if (!this.$timer)
            this.$timer = setTimeout(this.update, 100);
    };

    this.onMouseOut = function(e) {
        if (e && e.currentTarget.contains(e.relatedTarget))
            return;
        this.hide();
        this.editor.session.removeMarker(this.marker);
        this.$timer = clearTimeout(this.$timer);
    };

    this.setPosition = function(x, y) {
        if (x + 10 + this.width > this.maxWidth)
            x = window.innerWidth - this.width - 10;
        if (y > window.innerHeight * 0.75 || y + 20 + this.height > this.maxHeight)
            y = y - this.height - 30;

        Tooltip.prototype.setPosition.call(this, x + 10, y + 20);
    };

    this.destroy = function() {
        this.onMouseOut();
        aceEvent.removeListener(this.editor.renderer.scroller, "mousemove", this.onMouseMove);
        aceEvent.removeListener(this.editor.renderer.content, "mouseout", this.onMouseOut);
        delete this.editor.tokenTooltip;
    };

}).call(TokenTooltip.prototype);
